﻿using System;

namespace MVC_App.Validation
{
	public class ValidatingNode<TEntity>
	{
		public static implicit operator TEntity(ValidatingNode<TEntity> v)
		{
			if (v.DoEvaluate)
				return v.Value;

			v._errorWrapper.PrintErrors();
			return default(TEntity);
		}

		internal ValidatingNode(TEntity value, ErrorWrapper errorWrapper)
		{
			Value = value;
			_errorWrapper = errorWrapper;
		}

		internal ValidatingNode(ErrorWrapper errorWrapper)
		{
			Value = default(TEntity);
			_errorWrapper = errorWrapper;
		}

		internal readonly ErrorWrapper _errorWrapper;

		public TEntity Value { get; private set; }
		public bool DoEvaluate { get { return !_errorWrapper.HasError; } }

		#region Advanced

		public ValidatingNode<TResult> SelectMany<TNext, TResult>(
			Func<TEntity, ValidatingNode<TNext>> getNext,
			Func<TEntity, TNext, TResult> transformer)
		{
			var v2 = getNext(default(TEntity));
			var merge = _errorWrapper.Merge(v2._errorWrapper);
			if (DoEvaluate && v2.DoEvaluate)
			{
				try
				{
					var result = transformer(Value, v2.Value);
					return new ValidatingNode<TResult>(result, merge);
				}
				catch (ValidationException ve)
				{
					return new ValidatingNode<TResult>(merge.SetError(ve.ErrorMessage));
				}
			}
			return new ValidatingNode<TResult>(merge);
		}

		public ValidatingNode<TResult> SelectMany2<TNext, TResult>(
			Func<TEntity, ValidatingNode<TNext>> getNext,
			Func<TEntity, TNext, TResult> select)
		{
			// Die "from"-Variablen von einer Zeile sollen nicht in der nächsten Zeile
			// zugreifbar sein, deshalb geben wir hier explizit "default()" rein,
			// obwohl wir die Werte vielleicht haben. Aber eben nur vielleicht.
			// Bei anderen Monaden könnte dies völlig anders sein.
			var nextValidator = getNext(default(TEntity));
			if (DoEvaluate && nextValidator.DoEvaluate)
			{
				var result = select(Value, nextValidator.Value);
				return DoTransform(result);
			}
			return PropagateError<TResult>();
		}

		public ValidatingNode<TNew> TransformGuardNull<TNew>(Func<TEntity, TNew> transformer, string message)
			where TNew : class
		{
			if (DoEvaluate)
			{
				var result = transformer(Value);
				return result != null
					? DoTransform(result)
					: Error(message).PropagateError<TNew>();
			}
			return PropagateError<TNew>();
		}

		#endregion

		internal ValidatingNode<TNew> DoTransform<TNew>(TNew newValue)
		{
			if (DoEvaluate)
			{
				return new ValidatingNode<TNew>(newValue, _errorWrapper);
			}
			return PropagateError<TNew>();
		}

		public ValidatingNode<TNew> TransformGuardEx<TNew>(Func<TEntity, TNew> transformer, string message)
		{
			if (DoEvaluate)
			{
				try
				{
					return DoTransform(transformer(Value));
				}
				catch (Exception)
				{
					return Error(message).PropagateError<TNew>();
				}
			}
			return PropagateError<TNew>();
		}

		public ValidatingNode<TEntity> Error(string errorMessage)
		{
			return new ValidatingNode<TEntity>(_errorWrapper.SetError(errorMessage));
		}

		public ValidatingNode<TNew> PropagateError<TNew>()
		{
			return new ValidatingNode<TNew>(_errorWrapper);
		}
	}
}
